<?php

class MeowPWA_Core {
    public function init() {
        add_action('wp_head', array($this, 'add_pwa_tags'));
        add_action('init', array($this, 'handle_pwa_requests'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_pwa_scripts'));
    }

    public function add_pwa_tags() {
        $settings = get_option('meowpwa_settings', array());
        $theme_color = $settings['theme_color'] ?? '#ffffff';
        ?>
        <meta name="theme-color" content="<?php echo esc_attr($theme_color); ?>">
        <link rel="manifest" href="<?php echo home_url('/meowpwa-manifest.json'); ?>">
        <link rel="apple-touch-icon" href="<?php echo $this->get_icon_url(); ?>">
        <meta name="apple-mobile-web-app-capable" content="yes">
        <meta name="apple-mobile-web-app-status-bar-style" content="default">
        <meta name="apple-mobile-web-app-title" content="<?php echo esc_attr($settings['name'] ?? get_bloginfo('name')); ?>">
        <?php
    }

    public function enqueue_pwa_scripts() {
        wp_enqueue_script('meowpwa-main', MEOWPWA_URL . 'assets/js/main.js', array(), '1.0.0', true);
        wp_localize_script('meowpwa-main', 'meowpwa_vars', array(
            'sw_url' => home_url('/meowpwa-sw.js')
        ));
        wp_enqueue_style('meowpwa-style', MEOWPWA_URL . 'assets/css/style.css', array(), '1.0.0');
    }

    public function handle_pwa_requests() {
        $request = $_SERVER['REQUEST_URI'];
        
        if (strpos($request, 'meowpwa-manifest.json') !== false) {
            $this->serve_manifest();
            exit;
        }
        
        if (strpos($request, 'meowpwa-sw.js') !== false) {
            $this->serve_sw();
            exit;
        }
    }

    private function serve_manifest() {
        $settings = get_option('meowpwa_settings', array());
        $icon_url = $this->get_icon_url();
        
        $manifest = array(
            'name' => $settings['name'] ?? get_bloginfo('name'),
            'short_name' => $settings['short_name'] ?? get_bloginfo('name'),
            'description' => $settings['description'] ?? get_bloginfo('description'),
            'start_url' => home_url('/'),
            'display' => 'standalone',
            'theme_color' => $settings['theme_color'] ?? '#ffffff',
            'background_color' => $settings['background_color'] ?? '#ffffff',
            'icons' => array(
                array(
                    'src' => $icon_url,
                    'sizes' => '512x512',
                    'type' => 'image/png',
                    'purpose' => 'any maskable'
                )
            )
        );

        header('Content-Type: application/json');
        echo json_encode($manifest);
    }

    private function serve_sw() {
        $settings = get_option('meowpwa_settings', array());
        $offline_url = $settings['offline_page'] ? get_permalink($settings['offline_page']) : home_url('/');
        
        header('Content-Type: application/javascript');
        ?>
        const CACHE_NAME = 'meowpwa-cache-v1';
        const OFFLINE_URL = '<?php echo esc_url($offline_url); ?>';

        self.addEventListener('install', (event) => {
            event.waitUntil(
                caches.open(CACHE_NAME).then((cache) => {
                    return cache.addAll([OFFLINE_URL]);
                })
            );
        });

        self.addEventListener('fetch', (event) => {
            if (event.request.mode === 'navigate') {
                event.respondWith(
                    fetch(event.request).catch(() => {
                        return caches.match(OFFLINE_URL);
                    })
                );
            }
        });
        <?php
    }

    private function get_icon_url() {
        $settings = get_option('meowpwa_settings', array());
        if (!empty($settings['icon_id'])) {
            $url = wp_get_attachment_url($settings['icon_id']);
            if ($url) return $url;
        }
        return MEOWPWA_URL . 'assets/icon-512.png'; // 默认图标
    }
}
