<?php
/**
 * Plugin Name: MeowBOT
 * Description: 自动将登录、评论、文章操作及后台敏感设置修改推送至 Telegram Bot。
 * Version: 1.1.0
 * Author: HEMING.ORG
 */

if (!defined('ABSPATH')) {
    exit;
}

class MeowBOT {
    private $bot_token;
    private $chat_id;

    public function __construct() {
        // 获取配置
        $this->bot_token = get_option('meowbot_token', '');
        $this->chat_id = get_option('meowbot_chat_id', '');

        // 初始化钩子
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));

        // 1. 登录监控
        add_action('wp_login', array($this, 'log_user_login'), 10, 2);

        // 2. 文章操作监控
        add_action('transition_post_status', array($this, 'log_post_status_change'), 10, 3);
        add_action('before_delete_post', array($this, 'log_post_deletion'));

        // 3. 评论/留言监控
        add_action('comment_post', array($this, 'log_new_comment'), 10, 3);

        // 4. 后台敏感操作监控
        // 监控设置更新 (过滤非敏感选项以减少干扰)
        add_action('updated_option', array($this, 'log_option_update'), 10, 3);
        // 监控插件激活/禁用
        add_action('activated_plugin', array($this, 'log_plugin_activation'));
        add_action('deactivated_plugin', array($this, 'log_plugin_deactivation'));
        // 监控用户密码修改
        add_action('profile_update', array($this, 'log_profile_update'), 10, 2);
    }

    /**
     * 发送消息到 Telegram
     */
    public function send_to_telegram($message) {
        if (empty($this->bot_token) || empty($this->chat_id)) {
            return;
        }

        $url = "https://api.telegram.org/bot" . $this->bot_token . "/sendMessage";
        $data = array(
            'chat_id' => $this->chat_id,
            'text' => $message,
            'parse_mode' => 'HTML'
        );

        wp_remote_post($url, array(
            'body' => $data,
            'timeout' => 10,
        ));
    }

    /**
     * 1. 记录登录
     */
    public function log_user_login($user_login, $user) {
        $ip = $this->get_client_ip();
        $time = current_time('mysql');
        $msg = "<b>🔔 登录提醒</b>\n";
        $msg .= "用户: " . $user_login . "\n";
        $msg .= "IP: " . $ip . "\n";
        $msg .= "时间: " . $time;
        $this->send_to_telegram($msg);
    }

    /**
     * 2. 文章状态变化
     */
    public function log_post_status_change($new_status, $old_status, $post) {
        if ($post->post_type !== 'post') return;
        $time = current_time('mysql');
        $link = get_permalink($post->ID);
        $title = $post->post_title;

        if ($old_status !== 'publish' && $new_status === 'publish') {
            $msg = "<b>📝 新增文章</b>\n标题: " . $title . "\n链接: " . $link . "\n时间: " . $time;
            $this->send_to_telegram($msg);
        } elseif ($old_status === 'publish' && $new_status === 'publish') {
            $msg = "<b>✏️ 编辑文章</b>\n标题: " . $title . "\n链接: " . $link . "\n时间: " . $time;
            $this->send_to_telegram($msg);
        }
    }

    /**
     * 3. 文章删除
     */
    public function log_post_deletion($post_id) {
        $post = get_post($post_id);
        if (!$post || $post->post_type !== 'post') return;
        $time = current_time('mysql');
        $msg = "<b>🗑️ 删除文章</b>\n标题: " . $post->post_title . "\n时间: " . $time;
        $this->send_to_telegram($msg);
    }

    /**
     * 4. 新评论监控
     */
    public function log_new_comment($comment_ID, $comment_approved, $commentdata) {
        $time = current_time('mysql');
        $author = $commentdata['comment_author'];
        $content = wp_trim_words($commentdata['comment_content'], 20);
        $post_title = get_the_title($commentdata['comment_post_ID']);

        $msg = "<b>💬 新评论/留言</b>\n";
        $msg .= "来自: " . $author . "\n";
        $msg .= "文章: " . $post_title . "\n";
        $msg .= "内容: " . $content . "\n";
        $msg .= "时间: " . $time;
        $this->send_to_telegram($msg);
    }

    /**
     * 5. 敏感设置修改
     */
    public function log_option_update($option, $old_value, $value) {
        $sensitive_options = array('siteurl', 'home', 'admin_email', 'users_can_register', 'default_role', 'permalink_structure');
        if (!in_array($option, $sensitive_options)) return;

        $user = wp_get_current_user();
        $time = current_time('mysql');
        $msg = "<b>⚙️ 敏感设置修改</b>\n";
        $msg .= "操作者: " . $user->display_name . "\n";
        $msg .= "设置项: " . $option . "\n";
        $msg .= "新值: " . (is_array($value) ? json_encode($value) : $value) . "\n";
        $msg .= "时间: " . $time;
        $this->send_to_telegram($msg);
    }

    /**
     * 6. 插件操作
     */
    public function log_plugin_activation($plugin) {
        $user = wp_get_current_user();
        $msg = "<b>🔌 插件激活</b>\n操作者: " . $user->display_name . "\n插件: " . $plugin;
        $this->send_to_telegram($msg);
    }

    public function log_plugin_deactivation($plugin) {
        $user = wp_get_current_user();
        $msg = "<b>🔌 插件禁用</b>\n操作者: " . $user->display_name . "\n插件: " . $plugin;
        $this->send_to_telegram($msg);
    }

    /**
     * 7. 用户资料/密码修改
     */
    public function log_profile_update($user_id, $old_user_data) {
        $user = get_userdata($user_id);
        $current_user = wp_get_current_user();
        $time = current_time('mysql');
        
        $msg = "<b>👤 用户资料更新</b>\n";
        $msg .= "被修改用户: " . $user->user_login . "\n";
        $msg .= "操作者: " . $current_user->display_name . "\n";
        $msg .= "时间: " . $time;
        $this->send_to_telegram($msg);
    }

    private function get_client_ip() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        return $ip;
    }

    public function add_admin_menu() {
        add_options_page('MeowBOT 设置', 'MeowBOT', 'manage_options', 'meowbot', array($this, 'settings_page'));
    }

    public function register_settings() {
        register_setting('meowbot_group', 'meowbot_token');
        register_setting('meowbot_group', 'meowbot_chat_id');
    }

    public function settings_page() {
        ?>
        <div class="wrap">
            <h1>MeowBOT 设置</h1>
            <form method="post" action="options.php">
                <?php settings_fields('meowbot_group'); ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">Telegram Bot Token</th>
                        <td><input type="text" name="meowbot_token" value="<?php echo esc_attr(get_option('meowbot_token')); ?>" class="regular-text" /></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Telegram Chat ID</th>
                        <td><input type="text" name="meowbot_chat_id" value="<?php echo esc_attr(get_option('meowbot_chat_id')); ?>" class="regular-text" /></td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
            <hr>
            <p><strong>监控范围：</strong> 登录、新评论/留言、文章增删改、插件开关、关键站点设置修改、用户资料更新。</p>
        </div>
        <?php
    }
}

new MeowBOT();
